﻿///
/// パス文字列に対する処理関数を宣言するヘッダファイル。
///

#if       !defined(INCLUDED_SZ_PATH)
#define            INCLUDED_SZ_PATH

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

SZ_NS_BEG(szpp)

enum PathType
{
  PATH_TYPE_INVALID = 0,
  PATH_TYPE_RELATIVE,
  PATH_TYPE_DRIVE,
  PATH_TYPE_ROOT,
  PATH_TYPE_UNC,
  PATH_TYPE_QUALIFIED,
};

/// <summary>
/// パスの種類を返す関数。ディレクトリセパレータで終わる文字列だった場合は空文字列が返される。
/// </summary>
/// <param name="path">パス種別を判定するべきパス文字列。</param>
SZ_SPEC
PathType GetPathType(const szstring &path);

/// <summary>
/// 末端のファイル名を抽出する関数。ディレクトリセパレータで終わる文字列だった場合は空文字列が返される。
/// </summary>
/// <param name="path">ファイル名を抽出するべきパス文字列。</param>
SZ_SPEC
szstring ExtractFileName(const szstring &path);

/// <summary>
/// 末端のファイル名の拡張子を除いた部分を抽出する関数。
/// </summary>
/// <param name="path">抽出するべきパス文字列。</param>
SZ_SPEC
szstring ExtractStem(const szstring &path);

/// <summary>
/// 拡張子を抽出する関数。
/// </summary>
/// <param name="path">拡張子を抽出するべきパスまたはファイル名文字列。ディレクトリセパレータが含まれる文字列の場合も適切に処理される。</param>
/// <param name="inclusive">抽出する文字列にピリオドを含むなら true、含まないなら false を指定。</param>
SZ_SPEC
szstring ExtractExtension(const szstring &path, bool inclusive = false);

/// <summary>
/// ディレクトリ名を抽出する関数。最後のディレクトリセパレータまでの部分を取り出す。
/// </summary>
/// <param name="path">ディレクトリ名を抽出するべきパス文字列。</param>
/// <param name="inclusive">抽出する文字列にセパレータを含むなら true、含まないなら false を指定。</param>
SZ_SPEC
szstring ExtractDirectory(const szstring &path, bool inclusive = false);

/// <summary>
/// 拡張子部分を削除した文字列を返す関数。ただし最後のディレクトリセパレータより前にあるピリオドは拡張子区切りとはみなさない。
/// </summary>
/// <param name="path">拡張子部分を削除するべきパス文字列。</param>
SZ_SPEC
szstring RemoveExtension(const szstring &path);

/// <summary>
/// ファイル名部分を変更した文字列を返す関数。
/// </summary>
/// <param name="path">ファイル名部分を変更するべき元々のパス文字列。</param>
/// <param name="ext">新しいファイル名文字列。</param>
SZ_SPEC
szstring ChangeFileName(const szstring &path, const szstring &newName);

/// <summary>
/// 拡張子部分を変更した文字列を返す関数。
/// </summary>
/// <param name="path">拡張子部分を変更するべき元々のパス文字列。</param>
/// <param name="ext">新しい拡張子文字列（ピリオドは含まない）。</param>
SZ_SPEC
szstring ChangeExtension(const szstring &path, const szstring &ext);

/// <summary>
/// パスを二つつなげる関数。path1, path2 どちらもセパレータで終わらないこと。また、結合後のパスの末尾にもセパレータはつかない。
/// ただし path1EndsWithSep を true にすれば path1 の末尾にセパレータがついていてもよい。
/// </summary>
/// <param name="path1">つなげるパス文字列のうち前方のもの。</param>
/// <param name="path2">つなげるパス文字列のうち後方のもの。</param>
/// <param name="path1EndsWithSep">前者のパス文字列の末尾がディレクトリセパレータで終わるかどうかを表すフラグ。</param>
SZ_SPEC
szstring Combine(const szstring &path1, const szstring &path2, bool path1EndsWithSep = false);

/// <summary>
/// パス文字列の末尾からディレクトリセパレータを除去する関数。
/// </summary>
/// <param name="path">末尾からセパレータを取り除くパス文字列のポインタ。</param>
SZ_SPEC
void TrimSeparator(szstring *path);

/// <summary>
/// パスをディレクトリ／ファイル要素に分解する関数。
/// </summary>
/// <param name="path">分解するパス文字列。</param>
SZ_SPEC
std::deque<szstring> Decompose(const szstring &path);

/// <summary>
/// ディレクトリ／ファイル要素にパスに結合する関数。
/// </summary>
/// <param name="elements">結合するパス文字列要素の deque。</param>
SZ_SPEC
szstring Compose(const std::deque<szstring> &elements);

/// <summary>
/// ショートパス（実際にはロングパスであっても構わない）をロングパスに変換する関数。
/// </summary>
/// <param name="path">変換するパス文字列。</param>
SZ_SPEC
szstring ConvertToLongPath(const szstring &path);

/// <summary>
/// パスを正規化する関数。たとえばパスの途中に . があれば除去し、.. があれば親ディレクトリに上がる。
/// </summary>
/// <param name="path">正規化するパス文字列。</param>
SZ_SPEC
szstring Normalize(const szstring &path);

/// <summary>
/// 特定のディレクトリ以下に作成するテンポラリファイルのフルパスを構築する関数。
/// </summary>
/// <param name="dirPath">テンポラリファイルを作成するディレクトリのパス文字列。</param>
SZ_SPEC
szstring CreateTempFileName(const szstring &dirPath);

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_PATH)
